/*******************************************************************************
 * Macro Name: Amir_Mohammad_UHQ_Beam
 * Designer  : Amirhossein Ghadimi - Mohammad Bereyhi
 * Contacts	 : amir.ghadimi@epfl.ch - mjbereyhi@gmail.com
 *
 * 6 April 2018
 *******************************************************************************/
#include <cstdlib>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <cctype>

#include <string>


#define EXCLUDE_LEDIT_LEGACY_UPI

#define _USE_MATH_DEFINES 
#include <cmath>


#include <ldata.h>
#include "lcomp.h"

#define PI 3.14159
#define EE 2.7182818
#define ellipse_Number 400
#define Slice_Number 500
#define high_res_low_res_overlap 1000
#define high_res_pad_width 3000
#define trapizod_length 4e3



// All units are in nm


// This the main line which tells the L-Edit which functions to be called (L-Edit commands)
extern "C" {
	void UHQ_nanobeams(void);
	int UPI_Entry_Point(void);
}

//##########################################################
//##########################################################
//##########################################################
//###################### Basic Objects #####################
//##########################################################
//##########################################################
//##########################################################


//************************* Rectangle *************************************//
// This function draws rectangle in the given layer
// (x_position,y_position) is the position of the center of the rectangle
// x_width and y_height is the extension of the rectangle in x and y direction
void rectangle (float x_position,float y_position,float x_width,float y_height, char * layer ){
	
	// L-Edit commands, do not modify!
	LCell Cell_Draw = LCell_GetVisible ( );
	LFile File_Draw = LCell_GetFile ( Cell_Draw );
	
	// The LPolygon_New gets positions of the corners of a polygon and draw a polygon in the given layer
	LPoint RecPolygon[4];	// The positions of the polygon should store in an array of LPoint type (4 points in case of rectangle)
	
	// By function "LPoint_Set(x,y)" one can set the values of coordinates for each corner
	RecPolygon [0 ] = LPoint_Set (x_position-x_width/2,y_position-y_height/2);
	RecPolygon [1 ] = LPoint_Set (x_position+x_width/2,y_position-y_height/2);
	RecPolygon [2 ] = LPoint_Set (x_position+x_width/2,y_position+y_height/2);
	RecPolygon [3 ] = LPoint_Set (x_position-x_width/2,y_position+y_height/2);
	
	// Now we can draw our polygon by this function at the given layer
	LPolygon_New ( Cell_Draw, LLayer_Find ( File_Draw, layer ), RecPolygon, 4 );
		
}
//----------------------- End of rectangle ------------------------------//

//************************* rectangle_45_degree *************************************//
// This function draws 45 degree rotated rectangle in the given layer
// (x_position,y_position) is the position of the centre of the rectangle
// x_width and y_height is the extension of the rectangle in x and y direction
void rectangle_45_degree (float x_position,float y_position,float x_width,float y_height, char * layer ){
	
	// L-Edit commands, do not modify!
	LCell Cell_Draw = LCell_GetVisible ( );
	LFile File_Draw = LCell_GetFile ( Cell_Draw );
	
	// The LPolygon_New gets positions of the corners of a polygon and draw a polygon in the given layer
	LPoint RecPolygon[4];	// The positions of the polygon should store in an array of LPoint type (4 points in case of rectangle)
	
	// By function "LPoint_Set(x,y)" one can set the values of coordinates for each corner
	RecPolygon [0 ] = LPoint_Set (x_position-x_width*sqrt(2)/2-y_height*sqrt(2)/2,y_position+x_width*sqrt(2)/2-y_height*sqrt(2)/2);
	RecPolygon [1 ] = LPoint_Set (x_position-x_width*sqrt(2)/2+y_height*sqrt(2)/2,y_position+x_width*sqrt(2)/2+y_height*sqrt(2)/2);
	RecPolygon [2 ] = LPoint_Set (x_position+x_width*sqrt(2)/2+y_height*sqrt(2)/2,y_position-x_width*sqrt(2)/2+y_height*sqrt(2)/2);
	RecPolygon [3 ] = LPoint_Set (x_position+x_width*sqrt(2)/2-y_height*sqrt(2)/2,y_position-x_width*sqrt(2)/2-y_height*sqrt(2)/2);
	
	// Now we can draw our polygon by this function at the given layer
	LPolygon_New ( Cell_Draw, LLayer_Find ( File_Draw, layer ), RecPolygon, 4 );
		
}
//----------------------- End of rectangle_45_degree ------------------------------//

void trapezoid_pad_x (float x_position,float y_position,float x_width,float y_height_left,float y_height_right, char * layer ){
	
	// L-Edit commands, do not modify!
	LCell Cell_Draw = LCell_GetVisible ( );
	LFile File_Draw = LCell_GetFile ( Cell_Draw );
	
	// The LPolygon_New gets positions of the corners of a polygon and draw a polygon in the given layer
	LPoint RecPolygon[4];	// The positions of the polygon should store in an array of LPoint type (4 points in case of rectangle)
	
	// By function "LPoint_Set(x,y)" one can set the values of coordinates for each corner
	RecPolygon [0 ] = LPoint_Set (x_position-x_width/2,y_position-y_height_left/2);
	RecPolygon [1 ] = LPoint_Set (x_position+x_width/2,y_position-y_height_right/2);
	RecPolygon [2 ] = LPoint_Set (x_position+x_width/2,y_position+y_height_right/2);
	RecPolygon [3 ] = LPoint_Set (x_position-x_width/2,y_position+y_height_left/2);
	
	// Now we can draw our polygon by this function at the given layer
	LPolygon_New ( Cell_Draw, LLayer_Find ( File_Draw, layer ), RecPolygon, 4 );
		
}
//----------------------- End of trapezoid_pad_x ------------------------------//

void trapezoid_pad_y (float x_position,float y_position,float x_up,float x_down,float y_height, char * layer ){
	
	// L-Edit commands, do not modify!
	LCell Cell_Draw = LCell_GetVisible ( );
	LFile File_Draw = LCell_GetFile ( Cell_Draw );
	
	// The LPolygon_New gets positions of the corners of a polygon and draw a polygon in the given layer
	LPoint RecPolygon[4];	// The positions of the polygon should store in an array of LPoint type (4 points in case of rectangle)
	
	// By function "LPoint_Set(x,y)" one can set the values of coordinates for each corner
	RecPolygon [0 ] = LPoint_Set (x_position-x_down/2,y_position-y_height/2);
	RecPolygon [1 ] = LPoint_Set (x_position+x_down/2,y_position-y_height/2);
	RecPolygon [2 ] = LPoint_Set (x_position+x_up/2,y_position+y_height/2);
	RecPolygon [3 ] = LPoint_Set (x_position-x_up/2,y_position+y_height/2);
	
	// Now we can draw our polygon by this function at the given layer
	LPolygon_New ( Cell_Draw, LLayer_Find ( File_Draw, layer ), RecPolygon, 4 );
		
}
//----------------------- End of trapezoid_pad_y ------------------------------//


//****************************** Circle ********************************//
// This function draws a circle in the given layer
// (x_position,y_position) is the position of the centre of the circle
void circle (float x_position,float y_position,float radius, char * layer ){
	
	// L-Edit commands, do not modify!
	LCell Cell_Draw = LCell_GetVisible ( );
	LFile File_Draw = LCell_GetFile ( Cell_Draw );
	
	// L-Edit command to draw a circle in the given layer
	LCircle_New ( Cell_Draw, LLayer_Find ( File_Draw, layer ), LPoint_Set (x_position, y_position ), radius);
		
}
//--------------------------- End of Circle ---------------------------//


//************************* Ellipse *************************************//
// This function draws an ellipse in the given layer
// (x_position,y_position) is the position of the centre of the ellipse
// x_radius and y_radius are determining 2 radius of ellipse in x and y direction
void ellipse (float x_position,float y_position,float x_radius,float y_radius, char * layer ){
	
	// L-Edit commands, do not modify!
	LCell Cell_Draw = LCell_GetVisible ( );
	LFile File_Draw = LCell_GetFile ( Cell_Draw );
	
	
	// The LPolygon_New gets positions of the corners of a polygon and draw a polygon in the given layer
	LPoint ellipsePolygon[ellipse_Number];	// The positions of the polygon should store in an array of LPoint type (4 points in case of rectangle)
	
	float x=0;
	float y=0;
	
	// We approximate an ellipse by a "ellipse_Number" of segments which is calculate based on:
	for (int i=0; i<ellipse_Number+1; i++)
	{
		x=sin((2*PI/ellipse_Number)*i)*x_radius;
		y=cos((2*PI/ellipse_Number)*i)*y_radius;
		ellipsePolygon [i ] = LPoint_Set (x_position + x,y_position + y);
	}
	
	// Now we can draw our polygon by this function at the given layer
	LPolygon_New ( Cell_Draw, LLayer_Find ( File_Draw, layer ), ellipsePolygon, ellipse_Number );
		
}
//----------------------- End of ellipse ------------------------------//


//************************* text *************************************//
// This function draws text in the given layer
// (x_position,y_position) is the position of the begining of the text
// text_size determine the text size in nm units, lable is char* variable with the text that is intended to be written
void text (float x_position,float y_position,char * lable,float text_size, char * layer ){
	
	// L-Edit commands, do not modify!
	LCell Cell_Draw = LCell_GetVisible ( );
	LFile File_Draw = LCell_GetFile ( Cell_Draw );
	


	LCell_MakeLogo(Cell_Draw, lable, (LCoord)text_size, LLayer_Find ( File_Draw, layer ), LFALSE, LFALSE, LFALSE, (LCoord)x_position, (LCoord)y_position, LFALSE, LFALSE,LFALSE,NULL,NULL,NULL,NULL);	
	
	
}
//----------------------- End of text ------------------------------//

//##########################################################
//##########################################################
//##########################################################
//######## Basic Modules (Unit Cells + Beams + Etc.) #######
//##########################################################
//##########################################################
//##########################################################


//************************* Unit_Cell_Phononic_x *******************************//
// In this function we generate the Unit Cells of the phononic crystal (Big Piece + Small Piece) in x (Horizontal ) direction. 
// (x_position,y_position) is the coordinates of the unit cell
// direction=1 : small->left, direction=0: small-> right
void Unit_Cell_Phononic_x(float x_position,float y_position,float w_small,float l_small,float l_big,float w_big,bool direction,char *layer)
{
	if(direction==0)	// Right-> Small , Left-> big
	{
		// Here we draw the small piece:
		rectangle(x_position+(l_small/2)  ,y_position,l_small,w_small,layer);
		// Here we draw the big  piece:
		rectangle(x_position-(l_big/2)  ,y_position,l_big-trapizod_length,w_big,layer);
		trapezoid_pad_x ( x_position-(l_big), y_position, trapizod_length, w_small, w_big, layer );
		trapezoid_pad_x ( x_position, y_position, trapizod_length, w_big, w_small, layer );

	}
	else	// Right-> Big,  Left->  Small 
	{
		// Here we draw the small piece:
		rectangle(x_position-(l_small/2)  ,y_position,l_small,w_small,layer);
		// Here we draw the big  piece:
		rectangle(x_position+(l_big/2)  ,y_position,l_big-trapizod_length,w_big,layer);
		trapezoid_pad_x ( x_position+(l_big), y_position, trapizod_length, w_big, w_small, layer );
		trapezoid_pad_x ( x_position, y_position, trapizod_length, w_small, w_big, layer );
	}
}
//----------------------- End of Unit_Cell_Phononic_x ------------------------------//



//************************* Unit_Cell_Phononic_y *************************************//
// In this function we generate the Unit Cells of the phononic crystal (Big Piece + Small Piece) in y (vertical ) direction. 
// (x_position,y_position) is the coordinates of the unit cell
// direction=1 : small->up, direction=0: small-> down
void Unit_Cell_Phononic_y(float x_position,float y_position,float w_small,float l_small,float l_big,float w_big,bool direction,char *layer)
{
	if(direction==0)	// Right-> Small , Left-> big
	{
		// Here we draw the small piece:
		rectangle(x_position  ,y_position +(l_small/2),w_small ,l_small,layer);
		// Here we draw the big  piece:
		rectangle(x_position  ,y_position-(l_big/2),w_big ,l_big-trapizod_length,layer);
		trapezoid_pad_y (x_position  ,y_position-(l_big), w_big, w_small, trapizod_length, layer );
		trapezoid_pad_y (x_position  ,y_position, w_small, w_big, trapizod_length, layer );

	}
	else	// Right-> Big,  Left->  Small 
	{
		// Here we draw the small piece:
		rectangle(x_position  ,y_position-(l_small/2),w_small,l_small,layer);
		// Here we draw the big  piece:
		rectangle(x_position ,y_position+(l_big/2) ,w_big ,l_big-trapizod_length,layer);
		trapezoid_pad_y (x_position  ,y_position+(l_big), w_small, w_big, trapizod_length, layer );
		trapezoid_pad_y (x_position  ,y_position,  w_big,w_small, trapizod_length, layer );
	}
}
//----------------------- End of Unit_Cell_Phononic_y ------------------------------//

//************************* Unit_Cell_Phononic_inner_corougation_x *******************************//
// In this function we generate the Unit Cells of the phononic crystal (Big Piece + Small Piece) in x (Horizontal ) direction. 
// (x_position,y_position) is the coordinates of the unit cell
// direction=1 : small->left, direction=0: small-> right
void Unit_Cell_Phononic_inner_corougation_x(float x_position,float y_position,float w_small,float l_small,float l_big,float w_big,bool direction,char *layer)
{
	if(direction==0)	// Right-> Small , Left-> big
	{
		// Here we draw the small piece:
		rectangle(x_position+(l_small/2)  ,y_position-w_big/2+w_small/4,l_small,w_small/2,layer);
		rectangle(x_position+(l_small/2)  ,y_position+w_big/2-w_small/4,l_small,w_small/2,layer);
		// Here we draw the big  piece:
		rectangle(x_position-(l_big/2)  ,y_position,l_big-trapizod_length,w_big,layer);
		trapezoid_pad_x ( x_position-(l_big), y_position, trapizod_length, w_small, w_big, layer );
		trapezoid_pad_x ( x_position, y_position, trapizod_length, w_big, w_small, layer );

	}
	else	// Right-> Big,  Left->  Small 
	{
		// Here we draw the small piece:
		rectangle(x_position-(l_small/2)  ,y_position-w_big/2+w_small/4,l_small,w_small/2,layer);
		rectangle(x_position-(l_small/2)  ,y_position+w_big/2-w_small/4,l_small,w_small/2,layer);
		// Here we draw the big  piece:
		rectangle(x_position+(l_big/2)  ,y_position,l_big-trapizod_length,w_big,layer);
		trapezoid_pad_x ( x_position+(l_big), y_position, trapizod_length, w_big, w_small, layer );
		trapezoid_pad_x ( x_position, y_position, trapizod_length, w_small, w_big, layer );
	}
}
//----------------------- End of Unit_Cell_Phononic_inner_corougation_x ------------------------------//

//************************* Unit_Cell_Phononic_inner_corougation_y *************************************//
// In this function we generate the Unit Cells of the phononic crystal (Big Piece + Small Piece) in y (vertical ) direction. 
// (x_position,y_position) is the coordinates of the unit cell
// direction=1 : small->up, direction=0: small-> down
void Unit_Cell_Phononic_inner_corougation_y(float x_position,float y_position,float w_small,float l_small,float l_big,float w_big,bool direction,char *layer)
{
	if(direction==0)	// Right-> Small , Left-> big
	{
		// Here we draw the small piece:
		rectangle(x_position -w_big/2+w_small/4 ,y_position +(l_small/2),w_small/2 ,l_small,layer);
		rectangle(x_position +w_big/2-w_small/4 ,y_position +(l_small/2),w_small/2 ,l_small,layer);
		// Here we draw the big  piece:
		rectangle(x_position  ,y_position-(l_big/2),w_big ,l_big-trapizod_length,layer);
		trapezoid_pad_y (x_position  ,y_position-(l_big), w_big, w_small, trapizod_length, layer );
		trapezoid_pad_y (x_position  ,y_position, w_small, w_big, trapizod_length, layer );

	}
	else	// Right-> Big,  Left->  Small 
	{
		// Here we draw the small piece:
		rectangle(x_position -w_big/2+w_small/4 ,y_position -(l_small/2),w_small/2 ,l_small,layer);
		rectangle(x_position +w_big/2-w_small/4 ,y_position -(l_small/2),w_small/2 ,l_small,layer);		
		// Here we draw the big  piece:
		rectangle(x_position ,y_position+(l_big/2) ,w_big ,l_big-trapizod_length,layer);
		trapezoid_pad_y (x_position  ,y_position+(l_big), w_small, w_big, trapizod_length, layer );
		trapezoid_pad_y (x_position  ,y_position,  w_big,w_small, trapizod_length, layer );
	}
}
//----------------------- End of Unit_Cell_Phononic_inner_corougation_y ------------------------------//


//************************* Unit_Cell_Photonic_Circle *******************************//
// In this function we generate the Unit Cells of the Photonic crystal.  Rectangle and a hole inside it 
// (x_position,y_position) is the coordinates of the centre of the unit cell
// x_offset and y_offset determine the offset of the centre of the circle compare to rectangle 
void Unit_Cell_Photonic_Circle(float x_position,float y_position, float x_length ,float y_width, float radius, float x_offset, float y_offset)
{
	// Here we draw the rectangle:
	rectangle(x_position, y_position, x_length, y_width, "Mechanical_beam");
	
	// Now we draw the hole in it:
	circle ( x_position + x_offset, y_position + y_offset, radius, "Holes" );	
	
}
//----------------------- End of Unit_Cell_Photonic_Circle ------------------------------//


//************************* Unit_Cell_Photonic_Ellipes*******************************//
// In this function we generate the Unit Cells of the Photonic crystal.  Rectangle and a hole inside it 
// (x_position,y_position) is the coordinates of the centre of the unit cell
// x_radius and y_ radius determine the ellipses in x and y directions
// x_offset and y_offset determine the offset of the centre of the circle compare to rectangle 
void Unit_Cell_Photonic_Ellipes(float x_position,float y_position, float x_length ,float y_width, float x_radius,float y_radius, float x_offset, float y_offset)
{
	// Here we draw the rectangle:
	rectangle(x_position, y_position, x_length, y_width, "Mechanical_beam");
	
	// Now we draw the elliptical  hole in it:
	ellipse( x_position + x_offset, y_position + y_offset, x_radius, y_radius, "Holes");
	
}
//----------------------- End of Unit_Cell_Photonic_Ellipes ------------------------------//


//##########################################################
//##########################################################
//##########################################################
//###Devices and Packages (single beams disk + PC device)###
//##########################################################
//##########################################################
//##########################################################


//************************* Localized_beam_alone_x **************************//
// In this function we draw a localized beam with the pads and crystal (without any disk as a sensor) in x direction (horizontal)
// The input is the dimension and position of the device
// Type=0:  Phononic crystal is connected from small part to the beam. 
// Type=1: Phononic crystal is connected from big part to the beam. 
void Localized_beam_alone_x(float device_x_position,float device_y_position,float beam_width,float beam_length,
							 float w_small, float l_small,
							 float l_big, float w_big,int n_unit_cell, bool type,char *layer){
	
	// Note that the centre of the beam is centre of the device
	
	//We draw the beam in "Mechanical_beam" with above dimension and location
    rectangle(device_x_position  ,device_y_position, beam_length, beam_width, layer);

	// Now we draw n_unit_cell on the left and the right of the beam depends on the type of it
	float l_unit_cell=l_small+l_big;
	if(type==0)  // Phononic crystal is connected from small part to the beam. 
	{
		for(int i=0;i<n_unit_cell;i++)

		{
			Unit_Cell_Phononic_x(device_x_position + l_unit_cell*i + (l_small) +(beam_length/2) , device_y_position, w_small, l_small, l_big, w_big, 1,layer);
			Unit_Cell_Phononic_x(device_x_position - l_unit_cell*i - (l_small) -(beam_length/2) , device_y_position, w_small, l_small, l_big, w_big, 0,layer);
		}
	}
	else  // Phononic crystal is connected from big part to the beam. 
	{
		for(int i=0;i<n_unit_cell;i++)
		
		{
			Unit_Cell_Phononic_x(device_x_position + l_unit_cell*i + (l_big) +(beam_length/2) , device_y_position, w_small, l_small, l_big, w_big, 0,layer);
			Unit_Cell_Phononic_x(device_x_position - l_unit_cell*i - (l_big) -(beam_length/2) , device_y_position, w_small, l_small, l_big, w_big, 1,layer);
		}
	}
	

	}
//----------------------- End of Localized_beam_alone_x ------------------------//


//************************* Localized_beam_alone_y **************************//
//  In this function we draw a localized beam with the pads and crystal (without any disk as a sensor) in y direction (vertical)
// The input is the dimension and position of the device
// Type=0:  Phononic crystal is connected from small part to the beam. 
// Type=1: Phononic crystal is connected from big part to the beam. 
void Localized_beam_alone_y(float device_x_position,float device_y_position,float beam_width,float beam_length,
							 float w_small, float l_small,
							 float l_big, float w_big,int n_unit_cell, bool type,char *layer){
							 
	// Note that the centre of the beam is centre of the device
	
	//We draw the beam in "Mechanical_beam" with above dimension and location
    rectangle(device_x_position  ,device_y_position,beam_width ,beam_length,layer);

	
	// Now we draw n_unit_cell on top and bottom of the beam depends on the type of it
	float l_unit_cell=l_small+l_big;
	if(type==0)  // Phononic crystal is connected from small part to the beam. 
	{
		for(int i=0;i<n_unit_cell;i++)

		{
			Unit_Cell_Phononic_y(device_x_position  , device_y_position + l_unit_cell*i + (l_small) + (beam_length/2), w_small, l_small, l_big, w_big, 1,layer);
			Unit_Cell_Phononic_y(device_x_position  , device_y_position - l_unit_cell*i - (l_small) - (beam_length/2) , w_small, l_small, l_big, w_big, 0,layer);
		}
	}
	else  // Phononic crystal is connected from big part to the beam. 
	{
		for(int i=0;i<n_unit_cell;i++)
		
		{
			Unit_Cell_Phononic_y(device_x_position  , device_y_position + l_unit_cell*i + (l_big) +(beam_length/2), w_small, l_small, l_big, w_big, 0,layer);
			Unit_Cell_Phononic_y(device_x_position  , device_y_position - l_unit_cell*i - (l_big) -(beam_length/2) , w_small, l_small, l_big, w_big, 1,layer);
		}
	}
	
	
	
	}
//----------------------- End of Localized_beam_alone_y ------------------------//


//************************* stress_engineering_alone_x **************************//
// In this function we draw a tapered localized beam (stress engineering) in x direction (horizontal)
// The input is the dimension and position of the device
// Type=0:  Phononic crystal is connected from small part to the beam. 
// Type=1: Phononic crystal is connected from big part to the beam. 
void stress_engineering_alone_x(float device_x_position,float device_y_position,float defect_width,float defect_length,
							 float unitcell_width_ratio, float total_length,float DC_width_increase_factor,
							 float gaussin_width_n_unicell, float gaussian_amplitude,int n_unit_cell, bool type,char *layer){
	
	// Note that the centre of the beam is centre of the device
	

	// First we calculate the length and the width of each unicell in our tapered design
	float w_unicell_taper[100];
	float l_unit_cell_taper[100];
	
	// A gausiann envelope is appllied
	for (int i=1;i<n_unit_cell;i++)
	{
		w_unicell_taper[i]=(1/gaussian_amplitude)*(1-(1-gaussian_amplitude)*exp(-pow(((i-1)/gaussin_width_n_unicell),2)));
		l_unit_cell_taper[i]=1/sqrt(w_unicell_taper[i]);  // The unitcell length is swept in order to keep the banggap in the same position
	}
	// Then we hind the normalized length of the tapered design and scale it to the total length of our physical design
	float l_tot_normalized=0;
	for (int i=1;i<n_unit_cell;i++)
	{
		l_tot_normalized=l_tot_normalized +l_unit_cell_taper[i];
	}	
	float l_unitcell=total_length/(l_tot_normalized*2);
	
	// Then we patter the two chains in it size of the defect
	float unitcell_posittion=0;
	if(type==0)  // Phononic crystal is connected from small part to the beam. 
	{
		for(int i=0;i<n_unit_cell;i++)

		{
			Unit_Cell_Phononic_x(device_x_position + unitcell_posittion + l_unitcell*l_unit_cell_taper[i]/2 +(defect_length/2) , device_y_position, defect_width*w_unicell_taper[i]+DC_width_increase_factor, l_unitcell*l_unit_cell_taper[i]/2+4, l_unitcell*l_unit_cell_taper[i]/2, defect_width*w_unicell_taper[i]*unitcell_width_ratio+DC_width_increase_factor, 1,layer);
			Unit_Cell_Phononic_x(device_x_position - unitcell_posittion - l_unitcell*l_unit_cell_taper[i]/2 -(defect_length/2) , device_y_position, defect_width*w_unicell_taper[i]+DC_width_increase_factor, l_unitcell*l_unit_cell_taper[i]/2+4, l_unitcell*l_unit_cell_taper[i]/2, defect_width*w_unicell_taper[i]*unitcell_width_ratio+DC_width_increase_factor, 0,layer);
			unitcell_posittion = unitcell_posittion + l_unitcell*l_unit_cell_taper[i];
		}
	}
	else  // Phononic crystal is connected from big part to the beam. 
	{
		for(int i=0;i<n_unit_cell;i++)
		
		{
			Unit_Cell_Phononic_x(device_x_position + unitcell_posittion + l_unitcell*l_unit_cell_taper[i]/2 +(defect_length/2) , device_y_position, defect_width*w_unicell_taper[i]+DC_width_increase_factor, l_unitcell*l_unit_cell_taper[i]/2+4, l_unitcell*l_unit_cell_taper[i]/2, defect_width*w_unicell_taper[i]*unitcell_width_ratio+DC_width_increase_factor, 0,layer);
			Unit_Cell_Phononic_x(device_x_position - unitcell_posittion - l_unitcell*l_unit_cell_taper[i]/2 -(defect_length/2) , device_y_position, defect_width*w_unicell_taper[i]+DC_width_increase_factor, l_unitcell*l_unit_cell_taper[i]/2+4, l_unitcell*l_unit_cell_taper[i]/2, defect_width*w_unicell_taper[i]*unitcell_width_ratio+DC_width_increase_factor, 1,layer);
			unitcell_posittion = unitcell_posittion + l_unitcell*l_unit_cell_taper[i];
		}
	}
	
	//We draw the beam in "defect_beam" with above dimension and location
    rectangle(device_x_position  ,device_y_position, defect_length, defect_width+DC_width_increase_factor, layer);

	}
//----------------------- End of stress_engineering_alone_x ------------------------//

//************************* stress_engineering_alone_y **************************//
// In this function we draw a tapered localized beam (stress engineering) in y direction (horizontal)
// The input is the dimension and position of the device
// Type=0:  Phononic crystal is connected from small part to the beam. 
// Type=1: Phononic crystal is connected from big part to the beam. 
void stress_engineering_alone_y(float device_x_position,float device_y_position,float defect_width,float defect_length,
							 float unitcell_width_ratio, float total_length,float DC_width_increase_factor,
							 float gaussin_width_n_unicell, float gaussian_amplitude,int n_unit_cell, bool type,char *layer){
	
	// Note that the centre of the beam is centre of the device
	

	// First we calculate the length and the width of each unicell in our tapered design
	float w_unicell_taper[100];
	float l_unit_cell_taper[100];
	
	// A gausiann envelope is appllied
	for (int i=1;i<n_unit_cell;i++)
	{
		w_unicell_taper[i]=(1/gaussian_amplitude)*(1-(1-gaussian_amplitude)*exp(-pow(((i-1)/gaussin_width_n_unicell),2)));
		l_unit_cell_taper[i]=1/sqrt(w_unicell_taper[i]);  // The unitcell length is swept in order to keep the banggap in the same position
	}
	// Then we hind the normalized length of the tapered design and scale it to the total length of our physical design
	float l_tot_normalized=0;
	for (int i=1;i<n_unit_cell;i++)
	{
		l_tot_normalized=l_tot_normalized +l_unit_cell_taper[i];
	}	
	float l_unitcell=total_length/(l_tot_normalized*2);
	
	// Then we patter the two chains in it size of the defect
	float unitcell_posittion=0;
	if(type==0)  // Phononic crystal is connected from small part to the beam. 
	{
		for(int i=0;i<n_unit_cell;i++)

		{
			Unit_Cell_Phononic_y(device_x_position  , device_y_position + unitcell_posittion + l_unitcell*l_unit_cell_taper[i]/2 +(defect_length/2), defect_width*w_unicell_taper[i]+DC_width_increase_factor, l_unitcell*l_unit_cell_taper[i]/2, l_unitcell*l_unit_cell_taper[i]/2, defect_width*w_unicell_taper[i]*unitcell_width_ratio+DC_width_increase_factor, 1,layer);
			Unit_Cell_Phononic_y(device_x_position  , device_y_position - unitcell_posittion - l_unitcell*l_unit_cell_taper[i]/2 -(defect_length/2), defect_width*w_unicell_taper[i]+DC_width_increase_factor, l_unitcell*l_unit_cell_taper[i]/2, l_unitcell*l_unit_cell_taper[i]/2, defect_width*w_unicell_taper[i]*unitcell_width_ratio+DC_width_increase_factor, 0,layer);
			unitcell_posittion = unitcell_posittion + l_unitcell*l_unit_cell_taper[i];
		}
	}
	else  // Phononic crystal is connected from big part to the beam. 
	{
		for(int i=0;i<n_unit_cell;i++)
		
		{
			Unit_Cell_Phononic_y(device_x_position , device_y_position + unitcell_posittion + l_unitcell*l_unit_cell_taper[i]/2 +(defect_length/2) , defect_width*w_unicell_taper[i]+DC_width_increase_factor, l_unitcell*l_unit_cell_taper[i]/2, l_unitcell*l_unit_cell_taper[i]/2, defect_width*w_unicell_taper[i]*unitcell_width_ratio+DC_width_increase_factor, 0,layer);
			Unit_Cell_Phononic_y(device_x_position , device_y_position - unitcell_posittion - l_unitcell*l_unit_cell_taper[i]/2 -(defect_length/2) , defect_width*w_unicell_taper[i]+DC_width_increase_factor, l_unitcell*l_unit_cell_taper[i]/2, l_unitcell*l_unit_cell_taper[i]/2, defect_width*w_unicell_taper[i]*unitcell_width_ratio+DC_width_increase_factor, 1,layer);
			unitcell_posittion = unitcell_posittion + l_unitcell*l_unit_cell_taper[i];
		}
	}
	
	//We draw the beam in "defect_beam" with above dimension and location
    rectangle(device_x_position  ,device_y_position, defect_width+DC_width_increase_factor, defect_length, layer);

	}
//----------------------- End of stress_engineering_alone_y ------------------------//



//************************* LO_Window_pakage **************************//
// In this main function we put localized beam and a box around it
// direction=0 is x (horizontal) direction=1 is y (vertical)
void LO_Window_pakage(float pack_x_position,float pack_y_position, bool direction,
							 float beam_width,float beam_length,
							 float w_small, float l_small,
							 float l_big, float w_big,int n_unit_cell, bool type,char *layer,
							 float w_window){
	
	// if the direction is 0 is vertical, direction 1 is horizontal
	if (direction==0)
	{
	

	
	// First we draw a localized beam
	Localized_beam_alone_y( pack_x_position, pack_y_position, beam_width, beam_length,
							  w_small,  l_small,
							  l_big,  w_big, n_unit_cell,  type,layer);
	
	rectangle (pack_x_position,pack_y_position, w_window, beam_length + 2*n_unit_cell*(l_big+l_small), "Device_box" );

	}
	else
	{
	// First we draw a localized beam
	Localized_beam_alone_x( pack_x_position, pack_y_position, beam_width, beam_length,
							  w_small,  l_small,
							  l_big,  w_big, n_unit_cell,  type,layer);
							  
	rectangle (pack_x_position,pack_y_position, beam_length + 2*n_unit_cell*(l_big+l_small),w_window, "Device_box" );

	}

	
	}
//----------------------- End of LO_Window_pakage ------------------------//





//************************* LO_pad **************************//
// In this main function we put localized beam and a supporting pads
// direction=0 is x (horizontal) direction=1 is y (vertical)
void LO_pad(float pack_x_position,float pack_y_position, bool direction,
							 float beam_width,float beam_length,
							 float w_small, float l_small,
							 float l_big, float w_big,int n_unit_cell, bool type,char *layer,
							 float pad_x_width, float pad_y_height,char *pad_layer){
	
	// if the direction is 0 is vertical, direction 1 is horizontal
	if (direction==0)
	{
	

	
	// First we draw a localized beam
	Localized_beam_alone_y( pack_x_position, pack_y_position, beam_width, beam_length,
							  w_small,  l_small,
							  l_big,  w_big, n_unit_cell,  type,layer);
	
	rectangle (pack_x_position,pack_y_position+  beam_length/2 + n_unit_cell*(l_big+l_small)+(pad_x_width+high_res_pad_width)/2-high_res_low_res_overlap, pad_y_height, pad_x_width-high_res_pad_width , pad_layer );
	rectangle (pack_x_position,pack_y_position-  beam_length/2 - n_unit_cell*(l_big+l_small)-(pad_x_width+high_res_pad_width)/2+high_res_low_res_overlap, pad_y_height, pad_x_width-high_res_pad_width , pad_layer );

	rectangle (pack_x_position,pack_y_position+  beam_length/2 + n_unit_cell*(l_big+l_small)+high_res_pad_width/2,  pad_y_height , high_res_pad_width, layer );
	rectangle (pack_x_position,pack_y_position-  beam_length/2 - n_unit_cell*(l_big+l_small)-high_res_pad_width/2,  pad_y_height , high_res_pad_width, layer );

	
	}
	else
	{
	// First we draw a localized beam
	Localized_beam_alone_x( pack_x_position, pack_y_position, beam_width, beam_length,
							  w_small,  l_small,
							  l_big,  w_big, n_unit_cell,  type,layer);
							  
	rectangle (pack_x_position+  beam_length/2 + n_unit_cell*(l_big+l_small)+(pad_x_width+high_res_pad_width)/2-high_res_low_res_overlap,pack_y_position, pad_x_width-high_res_pad_width, pad_y_height , pad_layer );
	rectangle (pack_x_position-  beam_length/2 - n_unit_cell*(l_big+l_small)-(pad_x_width+high_res_pad_width)/2+high_res_low_res_overlap,pack_y_position, pad_x_width-high_res_pad_width, pad_y_height , pad_layer );

	rectangle (pack_x_position+  beam_length/2 + n_unit_cell*(l_big+l_small)+high_res_pad_width/2,pack_y_position, high_res_pad_width, pad_y_height , layer );
	rectangle (pack_x_position-  beam_length/2 - n_unit_cell*(l_big+l_small)-high_res_pad_width/2,pack_y_position, high_res_pad_width, pad_y_height , layer );
	
	}

	
	}
//----------------------- End of LO_pad ------------------------//

//************************* stress_engineering_pad **************************//
// In this main function we put tapered localized beam and a supporting pads
// direction=0 is x (horizontal) direction=1 is y (vertical)
void stress_engineering_pad(float pack_x_position,float pack_y_position, bool direction,
							 float beam_width,float beam_length,
							 float unitcell_width_ratio, float total_length, float DC_width_increase_factor,
							 float gaussin_width_n_unicell, float gaussian_amplitude,int n_unit_cell, bool type,char *layer,
							 float pad_x_width, float pad_y_height,char *pad_layer){
	
	// if the direction is 0 is vertical, direction 1 is horizontal
	if (direction==0)
	{
	
	// First we draw a localized beam
	stress_engineering_alone_y( pack_x_position, pack_y_position, beam_width, beam_length,
							  unitcell_width_ratio,  total_length, DC_width_increase_factor,
							  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type, layer);
	
	
	rectangle (pack_x_position,pack_y_position+  beam_length/2 + total_length/2 +(pad_x_width+high_res_pad_width)/2-high_res_low_res_overlap, pad_y_height, pad_x_width-high_res_pad_width , pad_layer );
	rectangle (pack_x_position,pack_y_position-  beam_length/2 - total_length/2 -(pad_x_width+high_res_pad_width)/2+high_res_low_res_overlap, pad_y_height, pad_x_width-high_res_pad_width , pad_layer );

	rectangle (pack_x_position,pack_y_position+  beam_length/2 + total_length/2 +high_res_pad_width/2,  pad_y_height , high_res_pad_width, layer );
	rectangle (pack_x_position,pack_y_position-  beam_length/2 - total_length/2 -high_res_pad_width/2,  pad_y_height , high_res_pad_width, layer );

	
	}
	else
	{
	// First we draw a localized beam
	stress_engineering_alone_x( pack_x_position, pack_y_position, beam_width, beam_length,
							  unitcell_width_ratio,  total_length, DC_width_increase_factor,
							  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type, layer);
							  
	rectangle (pack_x_position+  beam_length/2 + total_length/2 +(pad_x_width+high_res_pad_width)/2-high_res_low_res_overlap,pack_y_position, pad_x_width-high_res_pad_width, pad_y_height , pad_layer );
	rectangle (pack_x_position-  beam_length/2 - total_length/2 -(pad_x_width+high_res_pad_width)/2+high_res_low_res_overlap,pack_y_position, pad_x_width-high_res_pad_width, pad_y_height , pad_layer );

	rectangle (pack_x_position+  beam_length/2 + total_length/2 +high_res_pad_width/2,pack_y_position, high_res_pad_width, pad_y_height , layer );
	rectangle (pack_x_position-  beam_length/2 - total_length/2 -high_res_pad_width/2,pack_y_position, high_res_pad_width, pad_y_height , layer );
	
	}

	
	}

	

//##########################################################
//##########################################################
//##########################################################
//#####  Chip configuration (arrays, sweeps and etc.)#######
//##########################################################
//##########################################################
//##########################################################


//************************* chip_localized_defect_sweep   **************************//
//This is the main function where we build the chip made of the localized beam and the disk sensor
// This chip consist of number of block s and some number of devices in each block
// Inside each block we sweep the defect length with steps of "defect_sweep_step"
// Between blocks we sweep the offset between beam and disk with steps of "offset_sweep_step"
// All the structures are sitting on a long Mesa
void chip_localized_defect_sweep(float chip_x_position,float chip_y_position, float chip_x_width, float chip_y_height,
							float device_spacing, int n_devices, float defect_sweep_step, bool direction,
							float beam_width,float beam_length,
							 float w_small, float l_small,
							 float l_big, float w_big,int n_unit_cell, bool type,
							 float pad_x_width, float pad_y_height){
	

		char *layer="Mechanical_beam";
		char *pad_layer="Device_box";
		float width_increase_mesa=5e3;
		char beam_label[20];
		char temp_label[150];
		
		// Here we put an identifier which shows the wafer ID on the top of the chip
		text ( chip_x_position -2.3e6 , chip_y_position +chip_y_height/2 -500e3,"Ultra_Q_28(20nm)", 600e3, "MESA_layer_temp" );
		
		// Here we put an identifier which shows the sweep parameters of the chip	
		sprintf(temp_label,"N unitcell : %u",n_unit_cell);				
		text ( chip_x_position -2e6 , chip_y_position -chip_y_height/2 +900e3,temp_label, 300e3, "MESA_layer_temp" );
		sprintf(temp_label,"Width defect : %.0f",beam_width);						
		text ( chip_x_position -2e6 , chip_y_position -chip_y_height/2 +600e3,temp_label, 300e3, "MESA_layer_temp" );
		sprintf(temp_label,"Freq %.1f MHz",125e3/l_small);						
		text ( chip_x_position -2e6 , chip_y_position -chip_y_height/2 +300e3,temp_label, 300e3, "MESA_layer_temp" );		
		
		for (int i=0;i<n_devices+1;i++)	// This for controls the devices
		{
			// Depends on the direction , we change the position of the next device
			if (direction==1)
			{
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width, beam_length+ i*defect_sweep_step,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type, "Mechanical_beam",
								    pad_x_width,  pad_y_height,pad_layer);
									
			sprintf(beam_label,"%u",i+1);		
			
			text ( chip_x_position + (beam_length + i*defect_sweep_step)/2 + n_unit_cell*(l_small+l_big) + pad_x_width  + 300e3 , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,beam_label, 150e3, "MESA_layer_temp" );

									
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
					  beam_width+width_increase_mesa, beam_length+ i*defect_sweep_step,
					  w_small+width_increase_mesa,  l_small,
					  l_big,  w_big+width_increase_mesa, n_unit_cell,  type, "MESA_layer",
						pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");			
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
					  beam_width+width_increase_mesa, beam_length+ i*defect_sweep_step,
					  w_small+width_increase_mesa,  l_small,
					  l_big,  w_big+width_increase_mesa, n_unit_cell,  type, "MESA_layer",
						-width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");						
			}
			else
			{
			
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width, beam_length + i*defect_sweep_step,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type,"Mechanical_beam",
								   pad_x_width,  pad_y_height,pad_layer);

			sprintf(beam_label,"%u",i+1);		

			text ( chip_x_position + i*(device_spacing) - device_spacing*n_devices/2 -pad_x_width/2, chip_y_position  + (beam_length + i*defect_sweep_step)/2 + n_unit_cell*(l_small+l_big) + pad_y_height  + 300e3  ,beam_label, 130e3, "MESA_layer_temp" );

								   
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length + i*defect_sweep_step,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");									   
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length + i*defect_sweep_step,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");								   				   

			}		  
		}


		int i=n_devices+1;
		chip_y_height=chip_y_height+100e3;
		chip_x_width=chip_x_width+100e3;
		
			rectangle (chip_x_position , chip_y_position ,  chip_x_width , chip_y_height , "MESA_box" );



		
		// Puting a calibration wide beam at the end or at the beginning
		 i=-2;
		beam_width = 2.5e3;
		w_small=2.5e3;
		w_big =2.5e3;
		
			if (direction==1)
			{
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width, beam_length,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type, "Mechanical_beam",
								    pad_x_width,  pad_y_height,pad_layer);

			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width+width_increase_mesa, beam_length,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");	
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width+width_increase_mesa, beam_length,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");									   
			}
			else
			{
			
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width, beam_length ,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type, "Mechanical_beam",
								    pad_x_width,  pad_y_height,pad_layer);

			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length ,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");											
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length ,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");									   				   

			}		
	





	
		i=n_devices+1;
		n_unit_cell=0;
		beam_width = 2.5e3;
		beam_length= 1000e3;
		
			if (direction==1)
			{
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width, beam_length+ i*defect_sweep_step,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type,"Mechanical_beam",
								    pad_x_width,  pad_y_height,pad_layer);

			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width+width_increase_mesa, beam_length+ i*defect_sweep_step,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");	
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width+width_increase_mesa, beam_length+ i*defect_sweep_step,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");	
									}
			else
			{
			
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width, beam_length + i*defect_sweep_step,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type,"Mechanical_beam",
								   pad_x_width,  pad_y_height,pad_layer);
								   
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length + i*defect_sweep_step,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");								   				   
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length + i*defect_sweep_step,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");
			}	



	
	}
//----------------------- End of chip_localized_defect_sweep ------------------------//



//************************* chip_stress_engineering_localized_defect_sweep   **************************//
//This is the main function where we build the chip made of the localized beam and the disk sensor
// This chip consist of number of block s and some number of devices in each block
// Inside each block we sweep the defect length with steps of "defect_sweep_step"
// Between blocks we sweep the offset between beam and disk with steps of "offset_sweep_step"
// All the structures are sitting on a long Mesa
void chip_stress_engineering_localized_defect_sweep(float chip_x_position,float chip_y_position, float chip_x_width, float chip_y_height,
							float device_spacing, int n_devices, float defect_sweep_step, bool direction,
							 float beam_width,float beam_length,
							 float unitcell_width_ratio, float total_length,
							 float gaussin_width_n_unicell, float gaussian_amplitude,int n_unit_cell, bool type,
							 float pad_x_width, float pad_y_height){
	

		char *layer="Mechanical_beam";
		char *pad_layer="Device_box";
		float width_increase_mesa=1.5e3;
		float DC_width_increase_factor = 3e3;
		char beam_label[20];
		char temp_label[150];
		char* bias_layer="Bias_layer";
		


// First we calculate the length and the width of each unicell in our tapered design
	float w_unicell_taper[100];
	float l_unit_cell_taper[100];
	
	// A gausiann envelope is appllied
	for (int i=1;i<n_unit_cell;i++)
	{
		w_unicell_taper[i]=(1/gaussian_amplitude)*(1-(1-gaussian_amplitude)*exp(-pow(((i-1)/gaussin_width_n_unicell),2)));
		l_unit_cell_taper[i]=1/sqrt(w_unicell_taper[i]);  // The unitcell length is swept in order to keep the banggap in the same position
	}
	// Then we hind the normalized length of the tapered design and scale it to the total length of our physical design
	float l_tot_normalized=0;
	for (int i=1;i<n_unit_cell;i++)
	{
		l_tot_normalized=l_tot_normalized +l_unit_cell_taper[i];
	}	
	float l_unitcell=total_length/(l_tot_normalized*2);
	
	float first_unitcell_length=l_unit_cell_taper[1]*l_unitcell*2.2;
	
	

	int number_sweeps=7;
	float number_copies=n_devices/number_sweeps;

	float sweep_steps=first_unitcell_length*0.4/number_sweeps;

	float defect_lenght[200];
	
	int k=1;
	int sweep_step_loop_number=-3;
	for (int i=0;i<n_devices+1;i++)
	{
		if(k==number_copies+1)
		{
			k=1;	
			sweep_step_loop_number++;	
		}
		defect_lenght[i+1]=first_unitcell_length+sweep_step_loop_number*sweep_steps;
		k++;
	}

		
		
		for (int i=0;i<n_devices+1;i++)	// This for controls the devices
		{
			// Depends on the direction , we change the position of the next device
			if (direction==1)
			{
			stress_engineering_pad( chip_x_position,chip_y_position + i*(device_spacing) - device_spacing*n_devices/2,  direction,
								  beam_width, defect_lenght[i+1],
								  unitcell_width_ratio,  total_length,0,
								  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,layer,
								  pad_x_width,  pad_y_height, pad_layer);
			
			stress_engineering_pad( chip_x_position,chip_y_position + i*(device_spacing) - device_spacing*n_devices/2,  direction,
								  beam_width, defect_lenght[i+1],
								  unitcell_width_ratio,  total_length,DC_width_increase_factor,
								  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,"MESA_layer",
								  -width_increase_mesa,  pad_y_height+2*width_increase_mesa, "MESA_layer_temp");

			stress_engineering_pad( chip_x_position,chip_y_position + i*(device_spacing) - device_spacing*n_devices/2,  direction,
								  beam_width, defect_lenght[i+1],
								  unitcell_width_ratio,  total_length,DC_width_increase_factor,
								  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,"MESA_layer",
								  pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");								  
						
			}
			else
			{
			stress_engineering_pad( chip_x_position + i*(device_spacing) - device_spacing*n_devices/2,chip_y_position ,  direction,
					  beam_width, defect_lenght[i+1],
					  unitcell_width_ratio,  total_length,0,
					  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,layer,
					  pad_x_width,  pad_y_height, pad_layer);
			

			stress_engineering_pad( chip_x_position + i*(device_spacing) - device_spacing*n_devices/2,chip_y_position ,  direction,
								  beam_width, defect_lenght[i+1],
								  unitcell_width_ratio,  total_length,DC_width_increase_factor,
								  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,"MESA_layer",
								  -width_increase_mesa,  pad_y_height+2*width_increase_mesa, "MESA_layer_temp");
					  
			stress_engineering_pad( chip_x_position + i*(device_spacing) - device_spacing*n_devices/2,chip_y_position ,  direction,
								  beam_width, defect_lenght[i+1],
								  unitcell_width_ratio,  total_length,DC_width_increase_factor,
								  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,"MESA_layer",
								  pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa, "MESA_layer_temp");	
							
							   				   

			}		  
		}

		
		// Here we put an identifier which shows the wafer ID on the top of the chip
		text ( chip_x_position -2.3e6 , chip_y_position +chip_y_height/2 -500e3,"UQ43", 200e3, "Mechanical_beam" );
		
		// Here we put an identifier which shows the sweep parameters of the chip	
		
		sprintf(temp_label,"N: %u",n_unit_cell);				
		text ( chip_x_position -2e6 , chip_y_position -chip_y_height/2 +600e3,temp_label, 100e3, "Mechanical_beam" );
		sprintf(temp_label,"WD: %.0f nm",beam_width);						
		text ( chip_x_position -2e6 , chip_y_position -chip_y_height/2 +400e3,temp_label, 100e3, "Mechanical_beam" );
		sprintf(temp_label,"L: %.0f mm",total_length/1e6);						
		text ( chip_x_position -2e6 , chip_y_position -chip_y_height/2 +200e3,temp_label, 100e3, "Mechanical_beam" );	

		int i=n_devices+1;
		chip_y_height=chip_y_height+100e3;
		chip_x_width=chip_x_width+100e3;
		
			rectangle (chip_x_position , chip_y_position ,  chip_x_width , chip_y_height , "MESA_box" );
	
		


		
		// Puting a calibration wide beam at the end or at the beginning
		 i=-2;
		beam_width = 2.5e3;
		float w_small=2.5e3;
		float w_big =2.5e3;
		float l_big=total_length/(4*n_unit_cell);
		float l_small=total_length/(4*n_unit_cell);
		
			if (direction==1)
			{
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width, beam_length,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type, "Mechanical_beam",
								    pad_x_width,  pad_y_height,pad_layer);

			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width+width_increase_mesa, beam_length,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");	
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width+width_increase_mesa, beam_length,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");									   
			}
			else
			{
			
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width, beam_length ,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type, "Mechanical_beam",
								    pad_x_width,  pad_y_height,pad_layer);

			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length ,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");											
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length ,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");									   				   

			}		
	





	
		i=n_devices+1;
		n_unit_cell=0;
		beam_width = 2.5e3;
		beam_length= 1000e3;
		
			if (direction==1)
			{
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width, beam_length,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type,"Mechanical_beam",
								    pad_x_width,  pad_y_height,pad_layer);

			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width+width_increase_mesa, beam_length,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");	
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) - device_spacing*n_devices/2 ,  direction,
								  beam_width+width_increase_mesa, beam_length,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");	
									}
			else
			{
			
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width, beam_length ,
								  w_small,  l_small,
								  l_big,  w_big, n_unit_cell,  type,"Mechanical_beam",
								   pad_x_width,  pad_y_height,pad_layer);
								   
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length ,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   -width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");								   				   
			LO_pad( chip_x_position + i*(device_spacing)- device_spacing*n_devices/2, chip_y_position,  direction,
								  beam_width+width_increase_mesa, beam_length ,
								  w_small+width_increase_mesa,  l_small,
								  l_big,  w_big+width_increase_mesa, n_unit_cell,  type,"MESA_layer",
								   pad_x_width+2*width_increase_mesa,  pad_y_height+2*width_increase_mesa,"MESA_layer_temp");
			}	



	
	}
//----------------------- End of chip_stress_engineering_localized_defect_sweep ------------------------//


















//************************* chip_localized_n_unitcell_sweep   **************************//
//This is the main function where we build the chip made of the localized beam and the disk sensor
// This chip consist of number of block s and some number of devices in each block
// Inside each block we sweep the defect length with steps of "defect_sweep_step"
// Between blocks we sweep the offset between beam and disk with steps of "offset_sweep_step"
// All the structures are sitting on a long Mesa
void chip_localized_n_unitcell_sweep(float chip_x_position,float chip_y_position, float chip_x_width, float chip_y_height,
							float device_spacing, int n_devices, float defect_sweep_step, bool direction,
							float beam_width,float beam_length,
							 float w_small, float l_small,
							 float l_big, float w_big,int n_unit_cell,
							 int n_block, float block_spacing, bool type,
							 float pad_x_width, float pad_y_height){
	
		char *layer="Mechanical_beam";
		char *pad_layer="Device_box";

		
	for (int j=0;j<n_block;j++)
	{
		for (int i=0;i<n_unit_cell;i++)	// This for controls the devices
		{
			// Depends on the direction , we change the position of the next device
			if (direction==1)
			{
			LO_pad( chip_x_position , chip_y_position + i*(device_spacing) + j*(device_spacing*n_devices+block_spacing) - n_block*(device_spacing*n_devices+block_spacing)/2 ,  direction,
								  beam_width, beam_length+ j*defect_sweep_step,
								  w_small,  l_small,
								  l_big,  w_big, i,  type,layer,
								    pad_x_width,  pad_y_height,pad_layer);
			}
			else
			{
			
			LO_pad( chip_x_position + i*(device_spacing)+ j*(device_spacing*n_devices+block_spacing)- n_block*(device_spacing*n_devices+block_spacing)/2, chip_y_position,  direction,
								  beam_width, beam_length + j*defect_sweep_step,
								  w_small,  l_small,
								  l_big,  w_big, i,  type,layer,
								    pad_x_width,  pad_y_height,pad_layer);
								   
								   				   

			}		  
		}
	}
		



	
	}
//----------------------- End of chip_localized_n_unitcell_sweep ------------------------//



//##################################################################
//##################################################################
//##################################################################
//#######################  Alignment Marks  ########################
//##################################################################
//##################################################################
//##################################################################

//************************* pre_alignemnt_mark   **************************//
// This function draws the pre alignemnt marks for ebeam
void pre_alignemnt_mark(float pre_align_x_position,float pre_align_y_position){
	
	float box_size = 10e3;
	
	float space_base = 75e3;
	
	float space_increament = .5e3;
	
	for (int i=0; i<14;i++)
	{
		for (int j=0;j<14;j++)
		{
			// A cross shape alignment mark
			rectangle(pre_align_x_position  + i*(space_base + (i-1)*space_increament) , pre_align_y_position + j*(space_base + (j-1)*space_increament) , box_size  , box_size, "pre_alignment_layer");
			rectangle(pre_align_x_position  - i*(space_base + (i-1)*space_increament) , pre_align_y_position + j*(space_base + (j-1)*space_increament) , box_size  , box_size, "pre_alignment_layer");

			rectangle(pre_align_x_position  + i*(space_base + (i-1)*space_increament) , pre_align_y_position - j*(space_base + (j-1)*space_increament) , box_size , box_size , "pre_alignment_layer");
			rectangle(pre_align_x_position  - i*(space_base + (i-1)*space_increament) , pre_align_y_position - j*(space_base + (j-1)*space_increament) , box_size , box_size , "pre_alignment_layer");
		}
	}
	

	}
//----------------------- End of pre_alignemnt_mark ------------------------//


//************************* chip_aligment_mark   **************************//
// This function draws the alignment mark
void chip_aligment_mark(float align_x_position,float aligne_y_position){
	
	float alignment_width_cross=100e3;
	float alignment_length_cross=alignment_width_cross*1;
	float ebeam_alignment_box=10e3;
	float grading_thickness=2e3;
	float alignment_box = 4000e3;
	float box_spacing=300e3;
		
	for (int i=1;i<6;i++)
	{
		rectangle(align_x_position + alignment_length_cross/2 + i * box_spacing, aligne_y_position, ebeam_alignment_box,ebeam_alignment_box,"Chip_alignment_marks");
		rectangle(align_x_position - alignment_length_cross/2 - i * box_spacing, aligne_y_position, ebeam_alignment_box,ebeam_alignment_box,"Chip_alignment_marks");
		rectangle(align_x_position, aligne_y_position + alignment_length_cross/2 + i * box_spacing, ebeam_alignment_box,ebeam_alignment_box,"Chip_alignment_marks");
		rectangle(align_x_position, aligne_y_position - alignment_length_cross/2 - i * box_spacing, ebeam_alignment_box,ebeam_alignment_box,"Chip_alignment_marks");
	}
	
		

}

//************************* chip_aligment_mark   **************************//


//************************* dicing_aligment_mark   **************************//
// This function draws the alignment mark
void dicing_aligment_mark(float align_x_position,float aligne_y_position){
	
	float alignment_width_cross=50e3;
	float alignment_length_cross=alignment_width_cross*10;
	float ebeam_alignment_box=10e3;
	float grading_thickness=2e3;
	float alignment_box = 4000e3;
	float box_spacing=150e3;
	

	// A cross shape alignment mark
	rectangle(align_x_position  , aligne_y_position , alignment_width_cross,alignment_length_cross,"Chip_alignment_marks");
	rectangle(align_x_position  , aligne_y_position , alignment_length_cross,alignment_width_cross,"Chip_alignment_marks");

		

}

//************************* dicing_aligment_mark   **************************//


// This function draws the alignment mark
void aligment_mark(float align_x_position,float aligne_y_position){
	
	float alignment_width_cross=20e3;
	float alignment_length_cross=alignment_width_cross*10;
	float alignment_width_ebeam=20e3;
	float grading_thickness=2e3;
	float alignment_box = 3000e3;

	// A cross shape alignment mark
	rectangle(align_x_position  , aligne_y_position , alignment_width_cross,alignment_length_cross,"Photo_alignment_mark_cross");
	rectangle(align_x_position  , aligne_y_position , alignment_length_cross,alignment_width_cross,"Photo_alignment_mark_cross");
	
	// Grating parts
	for (int i=0; i<10; i++)
	{
		// In grating in y direction
		rectangle(align_x_position  , aligne_y_position + alignment_length_cross /2 + grading_thickness*(2*i+2) , alignment_width_cross,grading_thickness,"Photo_alignment_mark_cross");
		rectangle(align_x_position  , aligne_y_position - (alignment_length_cross /2 + grading_thickness*(2*i+2)) , alignment_width_cross,grading_thickness,"Photo_alignment_mark_cross");
		// In grating in x direction
		rectangle(align_x_position + alignment_length_cross /2 + grading_thickness*(2*i+2) , aligne_y_position  , grading_thickness, alignment_width_cross , "Photo_alignment_mark_cross");
		rectangle(align_x_position - (alignment_length_cross /2 + grading_thickness*(2*i+2)) , aligne_y_position  , grading_thickness, alignment_width_cross, "Photo_alignment_mark_cross");
	}
	
	// 45 degree cross 
	rectangle_45_degree ( align_x_position + alignment_box/4, aligne_y_position + alignment_box/4, grading_thickness, (alignment_box/2)/1.6, "Photo_alignment_mark_cross" );
	rectangle_45_degree ( align_x_position - alignment_box/4, aligne_y_position + alignment_box/4, (alignment_box/2)/1.6, grading_thickness, "Photo_alignment_mark_cross" );
	rectangle_45_degree ( align_x_position + alignment_box/4, aligne_y_position - alignment_box/4, (alignment_box/2)/1.6, grading_thickness, "Photo_alignment_mark_cross" );
	rectangle_45_degree ( align_x_position - alignment_box/4, aligne_y_position - alignment_box/4, grading_thickness, (alignment_box/2)/1.6, "Photo_alignment_mark_cross" );
	
	// 4 Rectangles alignment marks
	rectangle(align_x_position +2* alignment_width_cross, aligne_y_position + 2*alignment_width_cross, alignment_width_ebeam,alignment_width_ebeam,"Photo_alignment_mark_cross");
	rectangle(align_x_position -2* alignment_width_cross, aligne_y_position + 2*alignment_width_cross, alignment_width_ebeam,alignment_width_ebeam,"Photo_alignment_mark_cross");
	rectangle(align_x_position +2* alignment_width_cross, aligne_y_position - 2*alignment_width_cross, alignment_width_ebeam,alignment_width_ebeam,"Photo_alignment_mark_cross");
	rectangle(align_x_position -2* alignment_width_cross, aligne_y_position - 2*alignment_width_cross, alignment_width_ebeam,alignment_width_ebeam,"Photo_alignment_mark_cross");
	
	// A n empty box around the alignment box to make the search easier
	rectangle(align_x_position, aligne_y_position, alignment_box,alignment_box,"Photo_alignment_mark_box");


	}
//----------------------- End of alignment_mark ------------------------//




//##################################################################
//##################################################################
//##################################################################
//####################### Wafer structure   ########################
//##################################################################
//##################################################################
//##################################################################

//************************* Wafer   **************************//
// This function draws lines around the wafer and guide line of the cleaving
void Wafer(float wafer_x_position,float wafer_y_position, float wafer_radius,
			float chip_x_width, float chip_y_height, float line_width,float w_mesa, float h_mesa){
	

	int n_x=10;
	int n_y=4;
	float cond=0;
	float chip_holder_forbiden_zone=800e3;
	
	for (int i=-n_x;i<n_x+1;i++)
	{
		for(int j=-n_y;j<n_y+1;j++)
		{
			if(i*i<=49 && j*j<=9)
			{
			rectangle(wafer_x_position +(chip_x_width+line_width)*i  , wafer_y_position + (chip_y_height+line_width)*j , chip_x_width,chip_y_height,"Chip_design_layer");
			}
			
			rectangle(wafer_x_position +(chip_x_width+line_width)*i  , wafer_y_position + (chip_y_height+line_width)*j +(chip_y_height-chip_holder_forbiden_zone)/2, chip_x_width,chip_holder_forbiden_zone,"Chip_holder_forbiden_zone");
			rectangle(wafer_x_position +(chip_x_width+line_width)*i  , wafer_y_position + (chip_y_height+line_width)*j -(chip_y_height-chip_holder_forbiden_zone)/2, chip_x_width,chip_holder_forbiden_zone,"Chip_holder_forbiden_zone");

			if(j>-n_y && i<n_x+1 && i>-n_x)
			{
				chip_aligment_mark( wafer_x_position +(chip_x_width+line_width)*i -(chip_x_width+line_width)/2   , wafer_y_position + (chip_y_height+line_width)*j  -(chip_y_height+line_width)/2 );
			}
			cond=((i-.5)*5)*((i-.5)*5) +((j-.5)*12)*((j-.5)*12);
			if(cond> 35*35 && cond < 45*45)			
			{
				
					dicing_aligment_mark( wafer_x_position +(chip_x_width+line_width)*i -(chip_x_width+line_width)/2   , wafer_y_position + (chip_y_height+line_width)*j  -(chip_y_height+line_width)/2 );
				
			}
			
		}
	}
	
	
	pre_alignemnt_mark(wafer_x_position  -5* (chip_x_width+line_width) , wafer_y_position   -3 * (chip_y_height+line_width) );
	pre_alignemnt_mark(wafer_x_position  -5* (chip_x_width+line_width) , wafer_y_position   +3 * (chip_y_height+line_width) );
	pre_alignemnt_mark(wafer_x_position  +5* (chip_x_width+line_width) , wafer_y_position   +3 * (chip_y_height+line_width) );
	pre_alignemnt_mark(wafer_x_position  +5* (chip_x_width+line_width) , wafer_y_position   -3 * (chip_y_height+line_width) );
	


	
	
	for (int i=-1;i<=1;i++)
	{
		
			aligment_mark(  wafer_x_position + 8* (chip_x_width+line_width), wafer_y_position + i * (chip_y_height+line_width));
			aligment_mark(  wafer_x_position - 8* (chip_x_width+line_width), wafer_y_position + i * (chip_y_height+line_width));
		
	}
	
	
	
	
	// wafer boundaries
	rectangle(wafer_x_position , wafer_y_position -120e6/2+5e6 , 110.3e6,300e3,"Photo_alignment_mark_cross");
	rectangle(wafer_x_position , wafer_y_position +120e6/2-5e6 , 110.3e6,300e3,"Photo_alignment_mark_cross");
	rectangle(wafer_x_position -120e6/2+5e6 , wafer_y_position , 300e3,110.3e6,"Photo_alignment_mark_cross");
	rectangle(wafer_x_position +120e6/2-5e6 , wafer_y_position , 300e3,110.3e6,"Photo_alignment_mark_cross");

	rectangle(wafer_x_position , wafer_y_position -120e6/2 , 120.3e6,300e3,"Photo_alignment_mark_cross");
	rectangle(wafer_x_position , wafer_y_position +120e6/2 , 120.3e6,300e3,"Photo_alignment_mark_cross");
	rectangle(wafer_x_position -120e6/2 , wafer_y_position , 300e3,120.3e6,"Photo_alignment_mark_cross");
	rectangle(wafer_x_position +120e6/2 , wafer_y_position , 300e3,120.3e6,"Photo_alignment_mark_cross");

	rectangle(wafer_x_position , wafer_y_position -120e6/2+5e6 , 110.3e6,300e3,"MESA_layer");
	rectangle(wafer_x_position , wafer_y_position +120e6/2-5e6 , 110.3e6,300e3,"MESA_layer");
	rectangle(wafer_x_position -120e6/2+5e6 , wafer_y_position , 300e3,110.3e6,"MESA_layer");
	rectangle(wafer_x_position +120e6/2-5e6 , wafer_y_position , 300e3,110.3e6,"MESA_layer");

	rectangle(wafer_x_position , wafer_y_position -120e6/2 , 120.3e6,300e3,"MESA_layer");
	rectangle(wafer_x_position , wafer_y_position +120e6/2 , 120.3e6,300e3,"MESA_layer");
	rectangle(wafer_x_position -120e6/2 , wafer_y_position , 300e3,120.3e6,"MESA_layer");
	rectangle(wafer_x_position +120e6/2 , wafer_y_position , 300e3,120.3e6,"MESA_layer");
	
	
}
//----------------------- End of Wafer ------------------------//




//##########################################################
//##########################################################
//##########################################################
//########################  Main Function   ################
//##########################################################
//##########################################################
//##########################################################


//************************* main *************************************//
// Here is the main function which will be run by l-Edit
void UHQ_nanobeams(void){
	
	// L-Edit commands, do not modify!
	LC_InitializeState();
				
//----------------------------------------- Dimensions of  the wafer --------------------------------------//
	float wafer_x_position = 0;			// Wafer position x
	float wafer_y_position = 0;			// Wafer position y
	float wafer_radius    = 100e6;		// wafer radius (100mm)
	float wafer_radius_margin   = 2e6;  	

	bool direction = 1;					// direction of the devices and chips on the wafer
	
//----------------------------------------- Dimensions of  the chip  --------------------------------------//
	float chip_x_width = 5e6;		// Chip length
	float chip_y_height = 12e6;		// Chip width
	float line_width  = 200e3;		// Cleaving Line Thickness	
	
	// Mesa size
	float w_mesa = 400e3;
	float h_mesa = 8.9e6;

//--------------------------------------- Calibration beam dimensions --------------------------------------//	
	// Dimensions of  identical beam 
	float l_beam_ident	= 100000;	// Identical beam length
	float w_beam_ident	= 300;		// Identical beam width

//--------------------------------------- Optical Disk dimensions --------------------------------------//
	
	// Dimensions of ellipse
	float disk_radius_small = 14e3;	// Disks minimum radius
	float disk_radius_big = 36e3;	// Disks maximum radius

	float x_radius_step=1000;	// Radius Sweep Step in x direction
	float y_radius_step=2000;	// Radius Sweep Step in y direction	

	// Dimensions of circle
	float circle_radius = 25e3;	// Disks  radius
	
	float disk_offset = 000;	// Disk beam offset
	
//----------------------------------------- Localized beam dimensions --------------------------------------//
	// Dimensions of the beam(defect) and Pad
	float beam_width   = 500;		// Defect Beam width
	float beam_length  = 120e3;		// Defect Beam length 
	float pad_x_width  = 100e3;		// Pad width 
	float pad_y_height = 90e3;		// Pad height 
	
	// Dimensions of  phononic crystal
	float w_small	= 500;			// Small piece width
	float l_small	= 50e3 ;		// Small piece length
	float w_big		= 1000;			// Big  piece width
	float l_big  	= 50e3 ;		// Big piece length
	
	int n_unit_cell= 20;				// Number of unit cells in each side of the beam
	bool type =	1;					// Type of phononic crystal: Type=0:  Phononic crystal is connected from small part to the beam.  Type=1: Phononic crystal is connected from big part to the beam. 	

	float w_window = w_big + 20e3;	// Window width
	
	float triple_gap=5000;			// Gap between triple device
	
// --------------------- Special for localized beam and disk pack ---------------------//
	// In the block Parameters
	int n_devices=30;				// Number of devices in each block
	float device_spacing=120e3;		// Spacing in the block
	float defect_sweep_step=12e3/(n_devices*n_devices);	// Defect Sweep Step Size
	float radius_increment_step= 	6.5e3/(n_devices*n_devices);; // 2 Sweep the disk size with the beam

	

//-------------------- Special for paddle resonators ---------------------------//
	// Paddle parameters
	float paddle_width  = 30e3;	// Paddle Width
	float paddle_length = 30e3;	// Paddle Length
	float paddle_distance=5e3;	// Paddle distance in double paddle
	float paddle_width_sweep=500; //Paddle sweep steps
	
	// Support parameters
	float support_beam_width  = 100;	// Supporting beam wifth	
	float support_beam_length = 15e3;		// Supporting beam length 	
// ----------------------------- Special for all disk chip --------------------------------//
	float n_disk=20;	

// ----------------------------- Special for star shape anchor --------------------------------//
float l_anchor = 200e3;
float w_anchor = 1e3;
float angle_anchor=30*PI/180;
int n_anchor = 3;
float anchor_length_sweep_step = 20e3;

int n_block=5;
float block_spacing=200e3;



float defect_width=300;
float defect_length=50e3;

float unitcell_width_ratio = 2.3;
float total_length=3e6;
float gaussin_width_n_unicell=8;
float gaussian_amplitude=0.15;
 n_unit_cell=40;

float gaussin_width_n_unicell_optimum[]={8	,8	,8	,8	,9	,9	,9	,9	,9	,9	,10,	10,	10,	10,	10,	10,	10,	10,	11,	11,	11,	11,	11,	11,	11,	11,	11,	11,	12,	12,	12,	12,	12,	12,	12,	12,	12,	12,	12,	12,	12,	13,	13,	13,	13,	13,	13,	13,	13,	13,	13,	13,	13,	13,	13,	13,	13,	13,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	14,	15,	15,	15,	15,	15,	15,	15,	15,	15,	15,	15,	15,	15};
float gaussian_amplitude_optimum[]={0.29976057,	0.287094285,	0.275940709,	0.266018853,	0.257115998,	0.249067712,	0.241744585,	0.235043171,	0.228879637,	0.223185222,	0.217902913,	0.212984978,	0.208391111,	0.204087003,	0.200043237,	0.196234431,	0.192638547,	0.189236345,	0.186010942,	0.182947452,	0.180032691,	0.177254937,	0.174603724,	0.172069677,	0.169644366,	0.167320191,	0.165090277,	0.162948386,	0.160888848,	0.158906489,	0.156996584,	0.155154802,	0.153377168,	0.151660024,	0.15,	0.148393982,	0.146839091,	0.145332657,	0.143872204,	0.142455428,	0.141080185,	0.139744475,	0.138446431,	0.137184309,	0.135956474,	0.134761394,	0.133597632,	0.132463838,	0.131358743,	0.130281149,	0.129229931,	0.128204026,	0.127202429,	0.126224191,	0.125268415,	0.12433425,	0.123420891,	0.122527574,	0.121653573,	0.1207982,	0.119960799,	0.119140748,	0.118337453,	0.117550348,	0.116778896,	0.116022582,	0.115280916,	0.114553429,	0.113839673,	0.113139221,	0.112451661,	0.111776603,	0.111113671,	0.110462504,	0.109822758,	0.1091941,	0.108576214,	0.107968794,	0.107371547,	0.106784191,	0.106206455,	0.105638079,	0.105078811,	0.10452841,	0.103986643,	0.103453287,	0.102928125,	0.102410948,	0.101901556,	0.101399754,	0.100905356};

float support_pad_x_width=50e3;
float support_pad_y_height=25e3;
float support_length=30e3;
float support_width=50;
			 
////////////////////////
///// A group (1-13)////
////////////////////////
//In chip B1-13, we sweep the length defect inside chips and sweep the number of unitcells between chips


		direction=0;						 

	for(int i=0; i<13;i++)
	{

		total_length=7e6;
		n_unit_cell=2*i+17;
		beam_width=400;
		beam_length=(total_length/(2*n_unit_cell))*.1; //starts from 10% of the unitcell and sweep till 2.1 unitcell cites
		n_devices=35;
		defect_sweep_step= (total_length/(2*n_unit_cell))*2/n_devices;
		device_spacing=120e3;
								 
									  
									  
		chip_stress_engineering_localized_defect_sweep( wafer_x_position + (chip_x_width+line_width)*(i-6) , wafer_y_position  + (chip_y_height+line_width)*(2),  chip_x_width,  chip_y_height,
									 device_spacing,  n_devices,  defect_sweep_step,  direction,
									  beam_width, beam_length,
									  unitcell_width_ratio,  total_length,
									  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,
									  pad_x_width,  pad_y_height);									  

	}			

	
////////////////////////
///// B group (1-13)////
////////////////////////
//In chip B1-13, we sweep the length defect inside chips and sweep the number of unitcells between chips

		direction=1;						 

	for(int i=0; i<13;i++)
	{

		total_length=4e6;
		n_unit_cell=2*i+13;
		beam_width=400;
		beam_length=(total_length/(2*n_unit_cell))*.1; //starts from 10% of the unitcell and sweep till 2.1 unitcell cites
		n_devices=80;
		defect_sweep_step= (total_length/(2*n_unit_cell))*2/n_devices;
		device_spacing=120e3;
								 

		chip_stress_engineering_localized_defect_sweep( wafer_x_position + (chip_x_width+line_width)*(i-6) , wafer_y_position  + (chip_y_height+line_width)*(1),  chip_x_width,  chip_y_height,
									 device_spacing,  n_devices,  defect_sweep_step,  direction,
									  beam_width, beam_length,
									  unitcell_width_ratio,  total_length,
									  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,
									  pad_x_width,  pad_y_height);
	}											  

////////////////////////
///// C group (1-13)////
////////////////////////
//In chip B1-13, we sweep the length defect inside chips and sweep the number of unitcells between chips


		direction=1;						 

	for(int i=0; i<13;i++)
	{
		total_length=4e6;
		n_unit_cell=2*i+13;
		beam_width=400;
		beam_length=(total_length/(2*n_unit_cell))*.1; //starts from 10% of the unitcell and sweep till 2.1 unitcell cites
		n_devices=80;
		defect_sweep_step= (total_length/(2*n_unit_cell))*2/n_devices;
		device_spacing=120e3;
								 
									  
									  
		chip_stress_engineering_localized_defect_sweep( wafer_x_position + (chip_x_width+line_width)*(i-6) , wafer_y_position  + (chip_y_height+line_width)*(0),  chip_x_width,  chip_y_height,
									 device_spacing,  n_devices,  defect_sweep_step,  direction,
									  beam_width, beam_length,
									  unitcell_width_ratio,  total_length,
									  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,
									  pad_x_width,  pad_y_height);									  
	}						  							  
////////////////////////
///// D group (1-13)////
////////////////////////
//In chip B1-13, we sweep the length defect inside chips and sweep the number of unitcells between chips
	

		direction=0;						 
	

	for(int i=0; i<13;i++)
	{

		total_length=7e6;
		n_unit_cell=2*i+13;
		beam_width=400;
		beam_length=(total_length/(2*n_unit_cell))*.1; //starts from 10% of the unitcell and sweep till 2.1 unitcell cites
		n_devices=35;
		defect_sweep_step= (total_length/(2*n_unit_cell))*2/n_devices;
		device_spacing=120e3;
								 
									  
									  
		chip_stress_engineering_localized_defect_sweep( wafer_x_position + (chip_x_width+line_width)*(i-6) , wafer_y_position  + (chip_y_height+line_width)*(-1),  chip_x_width,  chip_y_height,
									 device_spacing,  n_devices,  defect_sweep_step,  direction,
									  beam_width, beam_length,
									  unitcell_width_ratio,  total_length,
									  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,
									  pad_x_width,  pad_y_height);									  

	}			

////////////////////////
///// E group (1-13)////
////////////////////////
//In chip B1-13, we sweep the length defect inside chips and sweep the number of unitcells between chips



		direction=1;						 

	for(int i=0; i<13;i++)
	{
		total_length=3e6;
		n_unit_cell=2*i+13;
		beam_width=400;
		beam_length=(total_length/(2*n_unit_cell))*.1; //starts from 10% of the unitcell and sweep till 2.1 unitcell cites
		n_devices=80;
		defect_sweep_step= (total_length/(2*n_unit_cell))*2/n_devices;
		device_spacing=120e3;
								 
									  
									  
		chip_stress_engineering_localized_defect_sweep( wafer_x_position + (chip_x_width+line_width)*(i-6) , wafer_y_position  + (chip_y_height+line_width)*(-2),  chip_x_width,  chip_y_height,
									 device_spacing,  n_devices,  defect_sweep_step,  direction,
									  beam_width, beam_length,
									  unitcell_width_ratio,  total_length,
									  gaussin_width_n_unicell,  gaussian_amplitude, n_unit_cell,  type,
									  pad_x_width,  pad_y_height);									  
	}						  			 
							 
	// Defining the alignment marks
	for (int i=-1;i<2;i++)
	{
		// Alignment Marks in Column 8
		aligment_mark( wafer_x_position +(chip_x_width+line_width)*8, wafer_y_position + (chip_y_height+line_width)*i);	
		aligment_mark( wafer_x_position -(chip_x_width+line_width)*8, wafer_y_position + (chip_y_height+line_width)*i);	

		}
	
	// Defining the wafer pattern 
	Wafer( wafer_x_position, wafer_y_position,  wafer_radius,
				 chip_x_width,  chip_y_height,  line_width, w_mesa,  h_mesa)	;	

				 
				 
}
//----------------------- End of main ------------------------------//

	// L-Edit commands, do not modify!
	// used to run the main function on L-Edit ("UHQ_nanobeams" here)
int UPI_Entry_Point(void)
{
	LMacro_BindToMenuAndHotKey_v9_30("Tools", NULL /*hotkey*/, 
			"UHQ_nanobeams", "UHQ_nanobeams", NULL /*hotkey category*/);
	return 1;
}
